<?php
defined( 'ABSPATH' ) || die();

require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_Helper.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Exam.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Question.php';

class OEM_Question {
	public static function fetch_questions() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		$exam_id = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

		global $wpdb;

		$page_url = OEM_Helper::get_page_url('exams');

		$query = OEM_M_Question::fetch_query( $exam_id );

		$query_filter = $query;

		// Grouping.
		$group_by = ' ' . OEM_M_Question::fetch_query_group_by();

		$query        .= $group_by;
		$query_filter .= $group_by;

		// Searching.
		$condition = '';
		if ( isset( $_POST['search']['value'] ) ) {
			$search_value = sanitize_text_field( $_POST['search']['value'] );
			if ( '' !== $search_value ) {
				$condition .= '' .
				'(q.type LIKE "%' . $search_value . '%") OR ' .
				'(q.negative_marks LIKE "%' . $search_value . '%") OR ' .
				'(q.maximum_marks LIKE "%' . $search_value . '%") OR ' .
				'(sj.subject_name LIKE "%' . $search_value . '%")';

				$query_filter .= ( ' HAVING ' . $condition );
			}
		}

		// Ordering.
		$columns = array( 'q.ID', 'q.type', 'q.maximum_marks', 'q.negative_marks', 'sj.subject_name', 'q.question_order' );
		if ( isset( $_POST['order'] ) && isset( $columns[ $_POST['order']['0']['column'] ] ) ) {
			$order_by  = sanitize_text_field( $columns[ $_POST['order']['0']['column'] ] );
			$order_dir = sanitize_text_field( $_POST['order']['0']['dir'] );

			// Sort by question order.
			if ( 'q.question_order' === $order_by ) {
				$order_by = 'q.question_order * 1';
			}

			$query_filter .= ' ORDER BY ' . $order_by . ' ' . $order_dir;
		} else {
			$query_filter .= ' ORDER BY sj.subject_order * 1 ASC, q.question_order * 1 ASC';
		}

		// Limiting.
		$limit = '';
		if ( -1 != $_POST['length'] ) {
			$start  = absint( $_POST['start'] );
			$length = absint( $_POST['length'] );

			$limit  = ' LIMIT ' . $start . ', ' . $length;
		}

		// Total query.
		$rows_query = OEM_M_Question::fetch_query_count( $exam_id );

		// Total rows count.
		$total_rows_count = $wpdb->get_var( $rows_query );

		// Filtered rows count.
		if ( $condition ) {
			$filter_rows_count = $wpdb->get_var( $rows_query . ' AND (' . $condition . ')' );
		} else {
			$filter_rows_count = $total_rows_count;
		}

		// Filtered limit rows.
		$filter_rows_limit = $wpdb->get_results( $query_filter . $limit );

		$data = array();
		if ( count( $filter_rows_limit ) ) {
			foreach ( $filter_rows_limit as $row ) {
				// Table columns.
				$data[] = array(
					esc_html( OEM_Helper::limit_string( strip_tags( $row->description ) ) ),
					esc_html( OEM_Helper::question_type( $row->type ) ),
					esc_html( OEM_Helper::sanitize_marks( $row->maximum_marks ) ),
					esc_html( OEM_Helper::sanitize_marks( $row->negative_marks, true, '-' ) ),
					esc_html( OEM_Helper::stripslashes( $row->subject_name, '-' ) ),
					esc_html( $row->question_order ),
					'<a class="oem-text-primary" href="' . esc_url( $page_url . "&action=questions&question_id=" . $row->ID . '&exam_id=' . $exam_id ) . '"><span class="dashicons dashicons-edit"></span></a>&nbsp;&nbsp;
					<a class="oem-text-danger oem-delete-question" data-nonce="' . esc_attr( wp_create_nonce( 'delete-question-' . $row->ID ) ) . '" data-question="' . esc_attr( $row->ID ) . '" data-exam="' . esc_attr( $row->exam_id ) . '" href="#" data-message-title="' . esc_attr__( 'Please Confirm!', 'online-exam-management' ) . '" data-message-content="' . esc_attr__( 'This will delete the question.', 'online-exam-management' ) . '" data-cancel="' . esc_attr__( 'Cancel', 'online-exam-management' ) . '" data-submit="' . esc_attr__( 'Confirm', 'online-exam-management' ) . '"><span class="dashicons dashicons-trash"></span></a>'
				);
			}
		}

		$output = array(
			'draw'            => absint( $_POST['draw'] ),
			'recordsTotal'    => $total_rows_count,
			'recordsFiltered' => $filter_rows_count,
			'data'            => $data,
		);

		echo json_encode( $output );
		die;
	}

	public static function save_question() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		$question_id = isset( $_POST['question_id'] ) ? absint( $_POST['question_id'] ) : 0;
		$exam_id     = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

		OEM_Helper::check_nonce( $question_id ? 'edit-question-' . $question_id : 'add-question' );

		try {
			ob_start();
			global $wpdb;

			$errors = array();

			// Checks if exam exists.
			$exam = OEM_M_Exam::get_exam( $exam_id );
			if ( ! $exam ) {
				throw new Exception( esc_html__( 'Exam not found.', 'online-exam-management' ) );
			}

			if ( $question_id ) {
				// Checks if question exists in exam.
				$question = OEM_M_Question::get_question( $question_id, $exam_id );
				if ( ! $question ) {
					throw new Exception( esc_html__( 'Question not found.', 'online-exam-management' ) );
				}
			}

			$subject_id     = isset( $_POST['subject_id'] ) ? absint( $_POST['subject_id'] ) : 0;
			$description    = isset( $_POST['description'] ) ? wp_kses_post( $_POST['description'] ) : '';
			$maximum_marks  = isset( $_POST['maximum_marks'] ) ? OEM_Helper::sanitize_marks( $_POST['maximum_marks'] ) : '';
			$negative_marks = isset( $_POST['negative_marks'] ) ? OEM_Helper::sanitize_marks( $_POST['negative_marks'] ) : 0;
			$question_order = isset( $_POST['question_order'] ) ? sanitize_text_field( $_POST['question_order'] ) : '';
			$type           = isset( $_POST['type'] ) ? sanitize_text_field( $_POST['type'] ) : '';

			if ( ! empty( $subject_id ) ) {
				$subject = OEM_M_Subject::get_subject( $subject_id, $exam_id );

				if ( ! $subject ) {
					$errors['subject_id'] = esc_html__( 'Please select valid subject.', 'online-exam-management' );
				}

			} else {
				// Assign default subject to this question.
				$default_subject = OEM_M_Exam::get_default_subject( $exam_id );
				if ( ! $default_subject ) {
					$errors['subject_id'] = esc_html__( 'Please select valid subject.', 'online-exam-management' );
				} else {
					$subject_id = $default_subject->ID;
				}
			}

			if ( empty( $description ) ) {
				$errors['description'] = esc_html__( 'Please specify question.', 'online-exam-management' );
			}

			if ( empty( $maximum_marks ) ) {
				$errors['maximum_marks'] = esc_html__( 'Please specify marks.', 'online-exam-management' );
			}

			if ( ! in_array( $type, array_keys( OEM_Helper::question_types() ) ) ) {
				$errors['type'] = esc_html__( 'Please select question type.', 'online-exam-management' );
			}

			if ( ! is_numeric( $question_order ) ) {
				$errors['question_order'] = esc_html__( 'Please provide question order.', 'online-exam-management' );
			}

			if ( count( $errors ) > 0 ) {
				wp_send_json_error( $errors );
			}

			if ( 'subjective' === $type ) {
				$negative_marks = 0;
				$options        = array();

			} else {
				$number_of_options  = isset( $_POST['number_of_options'] ) ? absint( $_POST['number_of_options'] ) : 0;
				$option             = ( isset( $_POST['option'] ) && is_array( $_POST['option'] ) ) ? array_map( 'wp_kses_post', $_POST['option'] ) : array();
				$existing_option_id = ( isset( $_POST['existing_option_id'] ) && is_array( $_POST['existing_option_id'] ) ) ? array_map( 'absint', $_POST['existing_option_id'] ) : array();


				if ( $number_of_options < 2 ) {
					throw new Exception( esc_html__( 'Please set atleast 2 options.', 'online-exam-management' ) );
				}

				if ( $number_of_options !== count( $option ) ) {
					throw new Exception( esc_html__( 'Please set correct number of options.', 'online-exam-management' ) );
				}

				if ( 'obj_multi' === $type ) {
					$correct_option = ( isset( $_POST['correct_option'] ) && is_array( $_POST['correct_option'] ) ) ? array_map( 'absint', $_POST['correct_option'] ) : array();
				} else {
					$correct_option = isset( $_POST['correct_option'] ) ? absint( $_POST['correct_option'] ) : '';
				}

				$options = array();

				$atleast_one_correct = false;
				for ( $i = 0; $i < $number_of_options; $i++ ) {
					$key        = $i;
					$is_correct = false;

					if ( ! isset( $option[ $key ] ) || empty( $option[ $key ] ) ) {
						throw new Exception(
							sprintf(
								/* translators: %s: Option number. */
								esc_html__( 'Please specify value for option %s.', 'online-exam-management' ),
								$key + 1
							)
						);
					}

					if ( 'obj_multi' === $type ) {
						if ( in_array( $key, $correct_option ) ) {
							$atleast_one_correct = true;
							$is_correct          = true;
						}

					} else {
						if ( $correct_option === $key ) {
							$atleast_one_correct = true;
							$is_correct          = true;
						}
					}

					array_push(
						$options,
						array(
							'value'      => $option[ $key ],
							'is_correct' => $is_correct
						)
					);
				}

				if ( ! $atleast_one_correct ) {
					if ( 'obj_multi' === $type ) {
						throw new Exception( esc_html__( 'Please check atleast one correct option.', 'online-exam-management' ) );
					} else {
						throw new Exception( esc_html__( 'Please select correct option.', 'online-exam-management' ) );
					}
				}
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			// Data to update or insert.
			$data = array(
				'description'    => $description,
				'maximum_marks'  => $maximum_marks,
				'negative_marks' => $negative_marks,
				'question_order' => $question_order,
				'type'           => $type,
				'subject_id'     => $subject_id,
				'exam_id'        => $exam_id,
			);

			// Checks if update or insert.
			if ( $question_id ) {
				$data['updated_at'] = OEM_Helper::now();

				$success = $wpdb->update( OEM_QUESTIONS, $data, array( 'ID' => $question_id ) );

				$message = esc_html__( 'Question updated successfully.', 'online-exam-management' );
				$reset   = false;

			} else {
				$data['created_at'] = OEM_Helper::now();

				$success     = $wpdb->insert( OEM_QUESTIONS, $data );
				$question_id = $wpdb->insert_id;

				$message = esc_html__( 'Question added successfully.', 'online-exam-management' );
				$reset   = true;
			}

			if ( count( $options ) ) {
				if ( ! empty( $existing_option_id ) ) {
					// Options exist for this question.
					$question_options = OEM_M_Question::get_options( $question_id );

					// Update existing options with new values.
					foreach ( $question_options as $key => $option ) {
						if ( in_array( $option->ID, $existing_option_id ) && isset( $options[ $key ] ) ) {

							$updated_option = $options[ $key ];

							$data = array(
								'value'      => $updated_option['value'],
								'is_correct' => $updated_option['is_correct']
							);

							$data['updated_at'] = OEM_Helper::now();

							$wpdb->update( OEM_OPTIONS, $data, array( 'ID' => $option->ID, 'question_id' => $question_id ) );
						}
					}

				} else {
					// Options do not exist for this question.
					$question_options_count = OEM_M_Question::get_options_count( $question_id );

					// Delete existing options for this question.
					if ( $question_options_count > 0 ) {
						$wpdb->delete( OEM_OPTIONS, array( 'question_id' => $question_id ) );
					}

					// Insert new options for this question.
					foreach ( $options as $key => $option ) {
						$data = array(
							'value'       => $option['value'],
							'question_id' => $question_id,
							'is_correct'  => $option['is_correct'],
						);

						$data['created_at'] = OEM_Helper::now();

						$wpdb->insert( OEM_OPTIONS, $data );
					}
				}

			} else {
				$wpdb->delete( OEM_OPTIONS, array( 'question_id' => $question_id ) );
			}

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message, 'reset' => $reset ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function delete_question() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		try {
			ob_start();
			global $wpdb;

			$exam_id     = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;
			$question_id = isset( $_POST['question_id'] ) ? absint( $_POST['question_id'] ) : 0;

			OEM_Helper::check_nonce( 'delete-question-' . $question_id );

			// Checks if question exists in exam.
			$question = OEM_M_Question::get_question( $question_id, $exam_id );

			if ( ! $question ) {
				throw new Exception( esc_html__( 'Question not found.', 'online-exam-management' ) );
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			$success = $wpdb->delete( OEM_QUESTIONS, array( 'ID' => $question_id ) );

			$message = esc_html__( 'Question deleted successfully.', 'online-exam-management' );

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function set_question_options() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		OEM_Helper::check_nonce('security');

		try {
			ob_start();

			$number_of_options = isset( $_POST['number_of_options'] ) ? absint( $_POST['number_of_options'] ) : 0;
			$question_type     = isset( $_POST['question_type'] ) ? sanitize_text_field( $_POST['question_type'] ) : '';

			for ( $i = 0; $i < $number_of_options; $i++ ) {
				$key        = $i;
				$value      = '';
				$is_correct = false;

				require OEM_PLUGIN_DIR_PATH . 'admin/inc/exams/fields/question/option.php';
			}

			$html = ob_get_clean();

			wp_send_json_success( array( 'html' => $html ) );

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function get_option( $option_id, $question_id ) {
		global $wpdb;

		return $wpdb->get_row(
			$wpdb->prepare( 'SELECT op.ID FROM ' . OEM_OPTIONS . ' as op WHERE op.ID = %d AND op.question_id = %d', $option_id, $question_id )
		);
	}

	public static function get_subjects( $exam_id ) {
		global $wpdb;

		return $wpdb->get_results(
			$wpdb->prepare( 'SELECT * FROM ' . OEM_SUBJECTS . ' as q 
			WHERE q.exam_id=%d', $exam_id )
		);
	}
}
