<?php
defined( 'ABSPATH' ) || die();

require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_Helper.php';
require_once OEM_PLUGIN_DIR_PATH . 'includes/helpers/OEM_M_Exam.php';

class OEM_Exam {
	public static function fetch_exams() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		global $wpdb;

		$page_url = OEM_Helper::get_page_url('exams');

		$query = OEM_M_Exam::fetch_query();

		$query_filter = $query;

		// Grouping.
		$group_by = ' ' . OEM_M_Exam::fetch_query_group_by();

		$query        .= $group_by;
		$query_filter .= $group_by;

		// Searching.
		$condition = '';
		if ( isset( $_POST['search']['value'] ) ) {
			$search_value = sanitize_text_field( $_POST['search']['value'] );
			if ( '' !== $search_value ) {
				$condition .= '' .
				'(ex.exam_title LIKE "%' . $search_value . '%") OR ' .
				'(ex.exam_code LIKE "%' . $search_value . '%") OR ' .
				'(ex.exam_center LIKE "%' . $search_value . '%") OR ' .
				'(ex.duration LIKE "%' . $search_value . '%")';

				$date_filter = OEM_Helper::filter_by_date( $search_value );

				$exam_at        = $date_filter['date'];
				$format_exam_at = $date_filter['format'];

				if ( $exam_at && ! empty( $format_exam_at ) ) {
					$exam_at = $exam_at->format( $format_exam_at );
					$exam_at = ' OR (ex.exam_at LIKE "%' . $exam_at . '%")';

					$condition .= $exam_at;
				}

				$query_filter .= ( ' HAVING ' . $condition );
			}
		}

		// Ordering.
		$columns = array( 'ex.exam_title', 'ex.exam_code', 'ex.exam_at', 'ex.duration', 'questions_count', 'maximum_marks' );
		if ( isset( $_POST['order'] ) && isset( $columns[ $_POST['order']['0']['column'] ] ) ) {
			$order_by  = sanitize_text_field( $columns[ $_POST['order']['0']['column'] ] );
			$order_dir = sanitize_text_field( $_POST['order']['0']['dir'] );

			$query_filter .= ' ORDER BY ' . $order_by . ' ' . $order_dir;
		} else {
			$query_filter .= ' ORDER BY ex.exam_at DESC';
		}

		// Limiting.
		$limit = '';
		if ( -1 != $_POST['length'] ) {
			$start  = absint( $_POST['start'] );
			$length = absint( $_POST['length'] );

			$limit  = ' LIMIT ' . $start . ', ' . $length;
		}

		// Total query.
		$rows_query = OEM_M_Exam::fetch_query_count();

		// Total rows count.
		$total_rows_count = $wpdb->get_var( $rows_query );

		// Filtered rows count.
		if ( $condition ) {
			$filter_rows_count = $wpdb->get_var( $rows_query . ' WHERE (' . $condition . ')' );
		} else {
			$filter_rows_count = $total_rows_count;
		}

		// Filtered limit rows.
		$filter_rows_limit = $wpdb->get_results( $query_filter . $limit );

		$data = array();
		if ( count( $filter_rows_limit ) ) {
			foreach ( $filter_rows_limit as $row ) {
				// Table columns.
				$data[] = array(
					esc_html( OEM_Helper::stripslashes( $row->exam_title ) ),
					esc_html( $row->exam_code ),
					esc_html( OEM_Helper::get_at_text( $row->exam_at ) ),
					absint( $row->duration ),
					absint( $row->questions_count ),
					esc_html( OEM_Helper::sanitize_marks( $row->maximum_marks ) ),
					OEM_Helper::get_exam_status_text( $row->exam_at, $row->duration, true ),
					'<a class="btn btn-sm oem-btn-success oem-font-bold" href="' . esc_url( $page_url . "&action=subjects&exam_id=" . $row->ID ) . '">' . esc_html__( 'Manage', 'online-exam-management' ) . '</a>',
					'<a class="btn btn-sm oem-btn-primary oem-font-bold" href="' . esc_url( $page_url . "&action=questions&exam_id=" . $row->ID ) . '">' . esc_html__( 'Manage', 'online-exam-management' ) . '</a>',
					'<a class="oem-text-primary" href="' . esc_url( $page_url . "&action=save&exam_id=" . $row->ID ) . '"><span class="dashicons dashicons-edit"></span></a>&nbsp;&nbsp;
					<a class="oem-text-danger oem-delete-exam" data-nonce="' . esc_attr( wp_create_nonce( 'delete-exam-' . $row->ID ) ) . '" data-exam="' . esc_attr( $row->ID ) . '" href="#" data-message-title="' . esc_attr__( 'Please Confirm!', 'online-exam-management' ) . '" data-message-content="' . esc_attr__( 'This will delete all data associated with this exam.', 'online-exam-management' ) . '" data-cancel="' . esc_attr__( 'Cancel', 'online-exam-management' ) . '" data-submit="' . esc_attr__( 'Confirm', 'online-exam-management' ) . '"><span class="dashicons dashicons-trash"></span></a>'
				);
			}
		}

		$output = array(
			'draw'            => absint( $_POST['draw'] ),
			'recordsTotal'    => $total_rows_count,
			'recordsFiltered' => $filter_rows_count,
			'data'            => $data,
		);

		echo json_encode( $output );
		die;
	}

	public static function save_exam() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		$exam_id = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

		OEM_Helper::check_nonce( $exam_id ? 'edit-exam-' . $exam_id : 'add-exam' );

		try {
			ob_start();
			global $wpdb;

			$errors = array();

			if ( $exam_id ) {
				// Checks if exam exists.
				$exam = OEM_M_Exam::get_exam( $exam_id );
				if ( ! $exam ) {
					throw new Exception( esc_html__( 'Exam not found.', 'online-exam-management' ) );
				}
			}

			$exam_title                = isset( $_POST['exam_title'] ) ? sanitize_text_field( $_POST['exam_title'] ) : '';
			$exam_code                 = isset( $_POST['exam_code'] ) ? sanitize_text_field( $_POST['exam_code'] ) : '';
			$exam_center               = isset( $_POST['exam_center'] ) ? sanitize_text_field( $_POST['exam_center'] ) : '';
			$exam_at                   = isset( $_POST['exam_at'] ) ? DateTime::createFromFormat( OEM_Helper::at_format(), sanitize_text_field( $_POST['exam_at'] ) ) : '';
			$duration                  = isset( $_POST['duration'] ) ? absint( $_POST['duration'] ) : '';
			$passing_percentage        = isset( $_POST['passing_percentage'] ) ? OEM_Helper::sanitize_marks( $_POST['passing_percentage'] ) : '';
			$instructions              = isset( $_POST['instructions'] ) ? wp_kses_post( $_POST['instructions'] ) : '';
			$is_randomized             = isset( $_POST['is_randomized'] ) ? (bool) $_POST['is_randomized'] : 0;
			$is_activated              = isset( $_POST['is_activated'] ) ? (bool) $_POST['is_activated'] : 0;
			$group_by_subject          = isset( $_POST['group_by_subject'] ) ? (bool) $_POST['group_by_subject'] : 0;
			$exam_fee                  = isset( $_POST['exam_fee'] ) ? OEM_Helper::sanitize_money( $_POST['exam_fee'] ) : 0;
			$show_in_payment_form      = isset( $_POST['show_in_payment_form'] ) ? (bool) $_POST['show_in_payment_form'] : 1;
			$show_result_on_completion = isset( $_POST['show_result_on_completion'] ) ? (bool) $_POST['show_result_on_completion'] : 0;
			$show_result_in_form       = isset( $_POST['show_result_in_form'] ) ? (bool) $_POST['show_result_in_form'] : 0;
			$roll_no_prefix            = isset( $_POST['roll_no_prefix'] ) ? sanitize_text_field( $_POST['roll_no_prefix'] ) : '';
			$roll_no_base              = isset( $_POST['roll_no_base'] ) ? absint( $_POST['roll_no_base'] ) : 0;
			$last_roll_no_index        = isset( $_POST['last_roll_no_index'] ) ? absint( $_POST['last_roll_no_index'] ) : 0;
			$exam_explain_link		   = isset( $_POST['exam_explain_link'] ) ? sanitize_text_field( $_POST['exam_explain_link'] ) : '';

			if ( empty( $exam_title ) ) {
				$errors['exam_title'] = esc_html__( 'Please specify exam title.', 'online-exam-management' );
			} elseif ( strlen( $exam_title ) > 191 ) {
				$errors['exam_title'] = esc_html__( 'Maximum length cannot exceed 191 characters.', 'online-exam-management' );
			}

			if ( empty( $exam_code ) ) {
				$errors['exam_code'] = esc_html__( 'Please specify exam code.', 'online-exam-management' );
			} elseif ( strlen( $exam_code ) > 255 ) {
				$errors['exam_code'] = esc_html__( 'Maximum length cannot exceed 255 characters.', 'online-exam-management' );
			}

			if ( ! empty( $exam_center ) && ( strlen( $exam_center ) > 255 ) ) {
				$errors['exam_center'] = esc_html__( 'Maximum length cannot exceed 255 characters.', 'online-exam-management' );
			}

			if ( empty( $exam_at ) ) {
				$errors['exam_at'] = esc_html__( 'Please specify exam date and time.', 'online-exam-management' );
			} else {
				$exam_at = $exam_at->format( 'Y-m-d H:i:s' );
			}

			if ( empty( $duration ) ) {
				$errors['duration'] = esc_html__( 'Please specify exam duration in minutes.', 'online-exam-management' );
			}

			if ( ! $passing_percentage ) {
				$passing_percentage = NULL;
			} elseif ( $passing_percentage > 100 ) {
				$errors['duration'] = esc_html__( "Passing percentage can't be greater than 100.", 'online-exam-management' );
			}

			if ( strlen( $roll_no_prefix ) > 15 ) {
				$errors['roll_no_prefix'] = esc_html__( 'Maximum length cannot exceed 25 characters.', 'online-exam-management' );
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		if ( count( $errors ) > 0 ) {
			wp_send_json_error( $errors );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			$page_url = OEM_Helper::get_page_url('exams');

			// Data to update or insert.
			$data = array(
				'exam_title'                => $exam_title,
				'exam_code'                 => $exam_code,
				'exam_center'               => $exam_center,
				'exam_at'                   => $exam_at,
				'duration'                  => $duration,
				'passing_percentage'        => $passing_percentage,
				'instructions'              => $instructions,
				'is_randomized'             => $is_randomized,
				'is_activated'              => $is_activated,
				'group_by_subject'          => $group_by_subject,
				'exam_fee'                  => $exam_fee,
				'show_in_payment_form'      => $show_in_payment_form,
				'show_result_on_completion' => $show_result_on_completion,
				'show_result_in_form'       => $show_result_in_form,
				'roll_no_prefix'            => $roll_no_prefix,
				'roll_no_base'              => $roll_no_base,
				'last_roll_no_index'        => $last_roll_no_index,
				'exam_explaination_link'	=> $exam_explain_link,
			);

			// Checks if update or insert.
			if ( $exam_id ) {
				$data['updated_at'] = OEM_Helper::now();

				$success = $wpdb->update( OEM_EXAMS, $data, array( 'ID' => $exam_id ) );

				$message = esc_html__( 'Exam updated successfully.', 'online-exam-management' );
				$url     = '';

			} else {
				$data['created_at'] = OEM_Helper::now();

				$success = $wpdb->insert( OEM_EXAMS, $data );
				$exam_id = $wpdb->insert_id;

				// Insert default subject for this exam.
				$subject_data = array(
					'subject_name'  => sanitize_text_field( OEM_Helper::get_default_subject() ),
					'subject_order' => 1,
					'is_default'    => 1,
					'exam_id'       => $exam_id,
					'created_at'    => OEM_Helper::now()
				);

				$success = $wpdb->insert( OEM_SUBJECTS, $subject_data );

				$message = esc_html__( 'Exam added successfully.', 'online-exam-management' );
				$url     = $page_url . '&action=save&exam_id=' . $exam_id;
			}

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message, 'url' => $url ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function delete_exam() {
		// OEM_Helper::check_demo_mode();

		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		try {
			ob_start();
			global $wpdb;

			$exam_id = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

			OEM_Helper::check_nonce( 'delete-exam-' . $exam_id );

			// Checks if exam exists.
			$exam = OEM_M_Exam::get_exam( $exam_id );
			if ( ! $exam ) {
				throw new Exception( esc_html__( 'Exam not found.', 'online-exam-management' ) );
			}

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			$success = $wpdb->delete( OEM_EXAMS, array( 'ID' => $exam_id ) );

			$message = esc_html__( 'Exam deleted successfully.', 'online-exam-management' );

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}

	public static function get_keyword_exams() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		OEM_Helper::check_nonce();

		$keyword = isset( $_POST['keyword'] ) ? sanitize_text_field( $_POST['keyword'] ) : '';

		$exam_id = isset( $_POST['exam'] ) ? absint( $_POST['exam'] ) : 0;

		$exams = OEM_M_Exam::get_exams_by_keyword( $keyword, $exam_id );

		$exams = array_map( function( $exam ) {
			$exam->label = esc_html( OEM_Helper::stripslashes( $exam->exam_title ) );
			unset( $exam->exam_title );

			if ( $exam->exam_code ) {
				$exam->label .= ' (' . esc_html( $exam->exam_code ) . ')';
				unset( $exam->exam_code );
			}

			$exam->label .= ' (' . esc_html( OEM_Helper::get_at_text( $exam->exam_at ) ) . ')';
			unset( $exam->exam_at );

			return $exam;
		}, $exams );

		wp_send_json_success( $exams );
	}

	public static function setup_activate() {
		if ( ! current_user_can( OEM_ADMIN_CAPABILITY ) ) {
			die();
		}

		try {
			ob_start();
			global $wpdb;

			$exam_id = isset( $_POST['exam_id'] ) ? absint( $_POST['exam_id'] ) : 0;

			OEM_Helper::check_nonce( 'setup-activate-exam-' . $exam_id );

			// Checks if exam exists.
			$exam = OEM_M_Exam::get_exam( $exam_id );
			if ( ! $exam ) {
				throw new Exception( esc_html__( 'Exam not found.', 'online-exam-management' ) );
			}

			$students = OEM_M_Student::fetch_exam_students_for_setup( $exam_id );

			OEM_Helper::check_buffer();

		} catch ( Exception $exception ) {
			wp_send_json_error( $exception->getMessage() );
		}

		try {
			$wpdb->query( 'BEGIN;' );

			foreach ( $students as $student ) {
				$success = $wpdb->update( OEM_STUDENTS, array( 'user_id' => NULL ), array( 'user_id' => $student->record_user_id ) );
				if ( false === $success ) {
					throw new Exception( $wpdb->last_error );
				}

				// Student data to update.
				$data = array(
					'user_id' => $student->record_user_id,
				);

				$success = $wpdb->update( OEM_STUDENTS, $data, array( 'ID' => $student->ID ) );

				OEM_Helper::check_buffer();

				if ( false === $success ) {
					throw new Exception( $wpdb->last_error );
				}
			}

			// Exam data to update.
			$data = array(
				'is_activated' => 1,
			);

			$success = $wpdb->update( OEM_EXAMS, $data, array( 'ID' => $exam_id ) );

			$message = esc_html__( 'Exam activated successfully.', 'online-exam-management' );

			OEM_Helper::check_buffer();

			if ( false === $success ) {
				throw new Exception( $wpdb->last_error );
			}

			$wpdb->query( 'COMMIT;' );

			wp_send_json_success( array( 'message' => $message ) );
		} catch ( Exception $exception ) {
			$wpdb->query( 'ROLLBACK;' );
			wp_send_json_error( $exception->getMessage() );
		}
	}
}
